/*!
 * Top mail — app.js
 * Enhancement #4:
 * - "add" now opens a modal pop-up form with vertical fields (no more inline add row)
 * Keeps:
 * - header-driven columns (prevents misalignment)
 * - status dropdown (status.php)
 * - grupo filter
 * - wrong disables send button
 * - private column only if present (admin)
 * - initials read-only unless admin
 * - cell editor overlay for long fields
 */

(function themeControl() {
  const THEMES = ['black', 'white', 'teal'];
  const THEME_COLORS = { black: '#0b0d12', white: '#ffffff', teal: 'rgb(36,183,175)' };

  function applyTheme(theme) {
    const b = document.body;
    b.classList.remove('theme-black', 'theme-white', 'theme-teal');
    b.classList.add('theme-' + theme);
    sessionStorage.setItem('topmail_theme', theme);

    const btn = document.getElementById('themeToggle');
    if (btn) {
      const idx = THEMES.indexOf(theme);
      const next = THEMES[(idx + 1) % THEMES.length];
      btn.style.background = THEME_COLORS[next];
      btn.setAttribute('data-next', next);
      btn.title = 'toggle background';
    }
  }

  applyTheme(sessionStorage.getItem('topmail_theme') || 'black');

  const themeBtn = document.getElementById('themeToggle');
  if (themeBtn) {
    themeBtn.addEventListener('click', function () {
      const cur = sessionStorage.getItem('topmail_theme') || 'black';
      const order = ['black', 'white', 'teal'];
      const next = order[(order.indexOf(cur) + 1) % order.length];
      applyTheme(next);
    });
  }
})();

(function layoutSizing() {
  function setSizeVars() {
    const topbar = document.querySelector('.topbar');
    const toolbar = document.querySelector('.toolbar');
    const topH = topbar ? topbar.offsetHeight : 52;
    const toolH = toolbar ? toolbar.offsetHeight : 28;
    document.documentElement.style.setProperty('--topbar-h', topH + 'px');
    document.documentElement.style.setProperty('--toolbar-h', toolH + 'px');
  }
  setSizeVars();
  window.addEventListener('resize', setSizeVars);
})();

(function scrollerGuard() {
  const wrap = document.querySelector('.table-wrap');
  if (!wrap) return;
  wrap.addEventListener(
    'wheel',
    function (e) {
      const horizontal = Math.abs(e.deltaX) > Math.abs(e.deltaY);
      if (!horizontal) return;
      const atLeft = wrap.scrollLeft <= 0;
      const atRight = Math.ceil(wrap.scrollLeft + wrap.clientWidth) >= wrap.scrollWidth;
      if ((e.deltaX < 0 && atLeft) || (e.deltaX > 0 && atRight)) {
        e.preventDefault();
        e.stopPropagation();
      }
    },
    { passive: false }
  );
})();

(function listPage() {
  const table = document.getElementById('companiesTable');
  if (!table) return;

  // Inject minimal modal styles (so no ceo.css change required)
  (function injectModalStyles(){
    const css = `
      .tm-backdrop{position:fixed;inset:0;background:rgba(0,0,0,.55);z-index:9998;display:flex;align-items:center;justify-content:center;padding:16px;}
      .tm-modal{width:min(720px,96vw);max-height:92vh;overflow:auto;background:var(--panel,#111);border:1px solid rgba(255,255,255,.12);border-radius:14px;box-shadow:0 18px 60px rgba(0,0,0,.45);padding:14px;}
      .tm-modal h3{margin:4px 0 10px 0;font-size:18px;opacity:.92;}
      .tm-grid{display:grid;grid-template-columns:1fr;gap:10px;}
      .tm-field label{display:block;font-size:12px;opacity:.8;margin-bottom:6px;}
      .tm-field input[type="text"], .tm-field select, .tm-field textarea{
        width:100%;box-sizing:border-box;padding:10px;border-radius:10px;border:1px solid rgba(255,255,255,.12);
        background:rgba(255,255,255,.04);color:inherit;outline:none;
      }
      .tm-field textarea{min-height:84px;resize:vertical;}
      .tm-row{display:flex;gap:10px;flex-wrap:wrap;align-items:center;justify-content:flex-end;margin-top:12px;}
      .tm-row .btn{white-space:nowrap;}
      .tm-inlinechecks{display:flex;gap:16px;flex-wrap:wrap;align-items:center;}
      .tm-inlinechecks label{display:flex;gap:8px;align-items:center;margin:0;opacity:.9;}
    `;
    const tag = document.createElement('style');
    tag.textContent = css;
    document.head.appendChild(tag);
  })();

  const wrap = document.querySelector('.table-wrap');
  const tbody = table.tBodies[0];

  const btnFilter = document.getElementById('btnFilter');
  const panel = document.getElementById('filterPanel');
  const countrySel = document.getElementById('countryFilter');
  const grupoSel = document.getElementById('grupoFilter');
  const applyBtn = document.getElementById('applyFilter');
  const clearBtn = document.getElementById('clearFilter');
  const btnAdd = document.getElementById('btnAdd');

  const qInput = document.getElementById('q');
  const btnClearSearch = document.getElementById('btnClearSearch');

  const urlParams = new URLSearchParams(location.search);
  const highlightId = urlParams.get('highlight_id');

  const IS_ADMIN = !!window.TOPMAIL_IS_ADMIN;

  const COLS = Array.from(table.querySelectorAll('thead th'))
    .map(th => (th.dataset.field || '').trim())
    .filter(Boolean);

  const EDITABLE = new Set([
    'company_name','ceo_name','MF','linkedIn','last_date','last_template',
    'country','language','history','email','sector','website','general_email','hq_city','comment',
    'grupo'
  ]);
  if (IS_ADMIN) EDITABLE.add('initials');

  const LONG_CLICK_TO_EXPAND = new Set([
    'history','comment','website','general_email','email','company_name','ceo_name','linkedIn'
  ]);

  // Counter UI
  let shownCountEl = null, totalCountEl = null;
  (function setupCountUI() {
    const toolbar = document.querySelector('.toolbar');
    if (!toolbar) return;
    const searchWrap = toolbar.querySelector('.search-wrap');
    const holder = document.createElement('div');
    holder.className = 'count-wrap';
    holder.style.marginLeft = '8px';
    holder.style.opacity = '0.85';
    holder.innerHTML = '<span id="shownCount">0</span> / <span id="totalCount">0</span>';
    if (searchWrap) searchWrap.insertAdjacentElement('afterend', holder);
    else toolbar.appendChild(holder);
    shownCountEl = holder.querySelector('#shownCount');
    totalCountEl = holder.querySelector('#totalCount');
  })();

  let STATUS_OPTIONS = [];
  async function fetchStatuses() {
    try {
      const r = await fetch('status.php');
      const d = await r.json();
      STATUS_OPTIONS = Array.isArray(d.statuses) ? d.statuses : [];
    } catch (_) {
      STATUS_OPTIONS = [];
    }
  }

  let state = {
    sortField: 'id',
    sortDir: 'ASC',
    country: 'all',
    grupo: 'all',
    q: '',
    rows: [],
    total: 0
  };

  function updateCount() {
    if (shownCountEl) shownCountEl.textContent = String(state.rows.length);
    if (totalCountEl) totalCountEl.textContent = String(state.total || 0);
  }

  function toggle(el, show) {
    if (!el) return;
    el.classList[show ? 'remove' : 'add']('hidden');
  }

  function dateOnly(str) {
    if (!str) return '';
    const m = String(str).match(/^(\d{4}-\d{2}-\d{2})/);
    return m ? m[1] : String(str);
  }

  function normalizeLinkedIn(raw) {
    let t = (raw || '').trim();
    if (!t) return '';
    if (/^https?:\/\/www\.linkedin\.com/i.test(t)) {
      t = t.replace(/^https?:\/\/www\.linkedin\.com/i, '');
      if (!t.startsWith('/')) t = '/' + t;
    }
    const q = t.indexOf('?');
    if (q !== -1) t = t.slice(0, q);
    return t;
  }

  function normalizeWebsite(raw) {
    let t = (raw || '').trim();
    if (!t) return '';
    t = t.replace(/^https?:\/\/(www\.)?/i, '');
    return t;
  }

  function setSendDisabledByWrong(tr, wrongVal) {
    const sendA = tr.querySelector('a.btn.small');
    if (!sendA) return;
    const bad = (String(wrongVal) === '1' || String(wrongVal).toLowerCase() === 'true');
    if (bad) {
      sendA.classList.add('ghost');
      sendA.style.pointerEvents = 'none';
      sendA.style.opacity = '0.45';
      sendA.title = 'Send disabled: email marked wrong';
    } else {
      sendA.classList.remove('ghost');
      sendA.style.pointerEvents = '';
      sendA.style.opacity = '';
      sendA.title = '';
    }
  }

  async function postUpdate(id, field, value) {
    await fetch('list.php?action=update', {
      method: 'POST',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
      body: new URLSearchParams({ id, field, value })
    });
  }

  async function fetchCountries() {
    const r = await fetch('list.php?action=countries');
    const d = await r.json();
    const countries = (d && d.countries) || [];
    const opts = ['all', ...countries];
    if (countrySel) {
      countrySel.innerHTML = opts.map(c => `<option value="${escapeAttr(c)}">${escapeHtml(c)}</option>`).join('');
      countrySel.value = state.country;
    }
  }

  async function fetchGrupos() {
    if (!grupoSel) return;
    const r = await fetch('list.php?action=grupos');
    const d = await r.json();
    const grupos = (d && d.grupos) || [];
    const opts = ['all', ...grupos];
    grupoSel.innerHTML = opts.map(g => `<option value="${escapeAttr(g)}">${escapeHtml(g)}</option>`).join('');
    grupoSel.value = state.grupo;
  }

  async function fetchRows() {
    const qs = new URLSearchParams({
      action: 'list',
      sort: state.sortField,
      dir: state.sortDir,
      country: state.country,
      grupo: state.grupo,
      q: state.q
    });
    const r = await fetch('list.php?' + qs.toString());
    const d = await r.json();

    state.rows = d.rows || [];
    if (typeof d.total === 'number') state.total = d.total;

    render();
    updateCount();

    if (highlightId) {
      const tr = Array.from(tbody.querySelectorAll('tr')).find((r) => {
        const lastIdCell = r.querySelector('td[data-field="id"]:last-child');
        return lastIdCell && lastIdCell.textContent.trim() === String(highlightId);
      });
      if (tr) {
        selectRow(tr);
        scrollRowIntoView(tr);
      }
    }
  }

  function render() {
    tbody.innerHTML = '';

    state.rows.forEach((row) => {
      const tr = document.createElement('tr');
      tr.dataset.rowId = row.id;

      COLS.forEach((col) => {
        if (col === 'actions') {
          const td = document.createElement('td');
          td.className = 'actions-cell';
          td.dataset.field = 'actions';

          const sendA = document.createElement('a');
          sendA.className = 'btn small';
          sendA.href = 'send.php?id=' + row.id;
          sendA.textContent = 'send';

          const linkBtn = document.createElement('button');
          linkBtn.className = 'btn small';
          linkBtn.textContent = 'link';
          linkBtn.dataset.action = 'link';
          if (!((row.linkedIn || '').trim())) {
            linkBtn.disabled = true;
            linkBtn.classList.add('ghost');
          }

          const delBtn = document.createElement('button');
          delBtn.className = 'btn small danger';
          delBtn.textContent = 'del';
          delBtn.dataset.action = 'del';

          td.appendChild(sendA);
          td.appendChild(linkBtn);
          td.appendChild(delBtn);
          tr.appendChild(td);

          setSendDisabledByWrong(tr, row.wrong);
          return;
        }

        if (col === 'status') {
          const td = document.createElement('td');
          td.dataset.field = 'status';
          td.classList.add('status-cell');

          const sel = document.createElement('select');
          sel.className = 'search-input';
          sel.style.maxWidth = '180px';

          const options = STATUS_OPTIONS.length ? STATUS_OPTIONS : [
            "Prospect","Email Sent","Follow-up","Meeting Booked","Qualified",
            "Proposal Sent","Negotiation","Closed won","Closed lost","Postponed"
          ];

          const cur = String(row.status || '').trim();

          const blank = document.createElement('option');
          blank.value = '';
          blank.textContent = '(blank)';
          if (cur === '') blank.selected = true;
          sel.appendChild(blank);

          options.forEach(v => {
            const opt = document.createElement('option');
            opt.value = v;
            opt.textContent = v;
            if (v === cur) opt.selected = true;
            sel.appendChild(opt);
          });

          sel.addEventListener('change', async () => {
            await postUpdate(row.id, 'status', sel.value);
          });

          td.appendChild(sel);
          tr.appendChild(td);
          return;
        }

        if (col === 'wrong') {
          const td = document.createElement('td');
          td.dataset.field = 'wrong';

          const cb = document.createElement('input');
          cb.type = 'checkbox';
          cb.checked = String(row.wrong || '0') === '1';
          cb.title = 'email wrong';
          cb.addEventListener('change', async () => {
            const v = cb.checked ? '1' : '0';
            await postUpdate(row.id, 'wrong', v);
            setSendDisabledByWrong(tr, v);
          });

          td.appendChild(cb);
          tr.appendChild(td);
          return;
        }

        if (col === 'private') {
          const td = document.createElement('td');
          td.dataset.field = 'private';

          const cb = document.createElement('input');
          cb.type = 'checkbox';
          cb.checked = String(row.private || '0') === '1';
          cb.title = 'private (admin only)';
          cb.addEventListener('change', async () => {
            const v = cb.checked ? '1' : '0';
            await postUpdate(row.id, 'private', v);
          });

          td.appendChild(cb);
          tr.appendChild(td);
          return;
        }

        const td = document.createElement('td');
        td.dataset.field = col;

        let val = (row[col] ?? '');
        if (col === 'last_date') val = dateOnly(val);

        td.textContent = val ?? '';
        if (val) td.title = String(val);

        if (col === 'initials' && !IS_ADMIN) {
          td.classList.add('readonly');
          tr.appendChild(td);
          return;
        }

        if (EDITABLE.has(col)) {
          td.contentEditable = true;
          td.classList.add('editable');
          td.dataset.id = row.id;

          if (LONG_CLICK_TO_EXPAND.has(col)) {
            td.addEventListener('click', openCellEditor);
            td.addEventListener('dblclick', openCellEditor);
          }
        }

        tr.appendChild(td);
      });

      tbody.appendChild(tr);
    });
  }

  function scrollRowIntoView(tr) {
    if (!tr || !wrap) return;
    wrap.scrollTo({ top: Math.max(tr.offsetTop - 4, 0), behavior: 'smooth' });
  }

  function clearSelection() {
    tbody.querySelectorAll('tr.row-selected').forEach((tr) => tr.classList.remove('row-selected'));
  }
  function selectRow(tr) {
    clearSelection();
    tr.classList.add('row-selected');
  }
  tbody.addEventListener('click', (e) => {
    const tr = e.target.closest('tr');
    if (tr) selectRow(tr);
  });

  // Inline saves on blur (for editable text cells)
  tbody.addEventListener('blur', function (e) {
    const td = e.target;
    if (!td.classList.contains('editable')) return;

    const id = td.dataset.id;
    const field = td.dataset.field;
    let value = (td.textContent || '').trim();

    if (field === 'linkedIn') value = normalizeLinkedIn(value);
    if (field === 'website') value = normalizeWebsite(value);

    td.textContent = value;
    td.title = value;

    postUpdate(id, field, value);

    if (field === 'linkedIn') {
      const btn = td.closest('tr')?.querySelector('button[data-action="link"]');
      if (btn) {
        const hasVal = value.length > 0;
        btn.disabled = !hasVal;
        btn.classList.toggle('ghost', !hasVal);
      }
    }
  }, true);

  // Live enable/disable link button while typing linkedin
  tbody.addEventListener('input', function (e) {
    const td = e.target;
    if (!td.classList.contains('editable')) return;
    if (td.dataset.field !== 'linkedIn') return;

    const btn = td.closest('tr')?.querySelector('button[data-action="link"]');
    if (!btn) return;

    const hasVal = (td.textContent || '').trim().length > 0;
    btn.disabled = !hasVal;
    btn.classList.toggle('ghost', !hasVal);
  });

  // Paste normalization for linkedin / website
  tbody.addEventListener('paste', async (e) => {
    const td = e.target.closest('td.editable');
    if (!td) return;

    const field = td.dataset.field;
    if (field !== 'linkedIn' && field !== 'website') return;

    const data = (e.clipboardData || window.clipboardData).getData('text');
    if (typeof data !== 'string') return;

    e.preventDefault();
    const id = td.dataset.id;

    const clean = field === 'linkedIn' ? normalizeLinkedIn(data) : normalizeWebsite(data);
    td.textContent = clean;
    td.title = clean;

    await postUpdate(id, field, clean);

    if (field === 'linkedIn') {
      const btn = td.closest('tr')?.querySelector('button[data-action="link"]');
      if (btn) {
        btn.disabled = !clean;
        btn.classList.toggle('ghost', !clean);
      }
    }
  });

  // Sorting
  table.querySelectorAll('th.sortable').forEach((th) => {
    th.addEventListener('click', () => {
      const f = th.dataset.field;
      if (!f || f === 'actions') return;
      if (state.sortField === f) state.sortDir = (state.sortDir === 'ASC') ? 'DESC' : 'ASC';
      else { state.sortField = f; state.sortDir = 'ASC'; }
      fetchRows();
    });
  });

  // Filter panel
  if (btnFilter) btnFilter.addEventListener('click', () => toggle(panel, panel.classList.contains('hidden')));
  if (applyBtn) {
    applyBtn.addEventListener('click', () => {
      state.country = countrySel ? countrySel.value : 'all';
      state.grupo = grupoSel ? grupoSel.value : 'all';
      fetchRows();
      toggle(panel, false);
    });
  }
  if (clearBtn) {
    clearBtn.addEventListener('click', () => {
      state.country = 'all';
      state.grupo = 'all';
      if (countrySel) countrySel.value = 'all';
      if (grupoSel) grupoSel.value = 'all';
      fetchRows();
      toggle(panel, false);
    });
  }

  // Search
  let searchTimer = null;
  function triggerSearch() {
    state.q = (qInput && qInput.value) ? qInput.value.trim() : '';
    fetchRows();
  }
  if (qInput) {
    qInput.addEventListener('input', function () {
      clearTimeout(searchTimer);
      searchTimer = setTimeout(triggerSearch, 400);
    });
    qInput.addEventListener('keydown', function (e) {
      if (e.key === 'Enter') { e.preventDefault(); clearTimeout(searchTimer); triggerSearch(); }
      if (e.key === 'Escape') { qInput.value = ''; clearTimeout(searchTimer); triggerSearch(); }
    });
  }
  if (btnClearSearch) {
    btnClearSearch.addEventListener('click', function () {
      if (qInput) qInput.value = '';
      state.q = '';
      fetchRows();
    });
  }

  // Actions: link, del (delete is admin-only server-side)
  tbody.addEventListener('click', async (e) => {
    const btn = e.target.closest('button[data-action]');
    if (!btn) return;

    const tr = btn.closest('tr');
    if (!tr) return;

    const id = tr.dataset.rowId;
    const action = btn.dataset.action;

    if (action === 'link') {
      const linkedInCell = tr.querySelector('td[data-field="linkedIn"]');
      const companyCell = tr.querySelector('td[data-field="company_name"]');

      const liRaw = linkedInCell ? linkedInCell.textContent.trim() : '';
      const coRaw = companyCell ? companyCell.textContent.trim() : '';

      if (!liRaw) {
        const q = encodeURIComponent(coRaw || '');
        window.open('https://www.linkedin.com/search/results/companies/?keywords=' + q, '_blank');
        return;
      }

      let li = liRaw;
      if (!/^https?:\/\//i.test(li)) li = 'https://www.linkedin.com' + (li.startsWith('/') ? li : '/' + li);
      window.open(li, '_blank');
      return;
    }

    if (action === 'del') {
      if (!btn.dataset.confirming) {
        btn.dataset.confirming = '1';
        btn.dataset.origText = btn.textContent;
        btn.textContent = '✓';

        const cancel = document.createElement('button');
        cancel.className = 'btn small ghost';
        cancel.textContent = '✕';
        cancel.dataset.action = 'del_cancel';
        btn.insertAdjacentElement('afterend', cancel);
        return;
      }

      const res = await fetch('list.php?action=delete', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({ id })
      });
      const data = await res.json();
      if (!data.ok) alert(data.error || 'delete failed');

      await fetchCountries();
      await fetchGrupos();
      await fetchRows();
      return;
    }

    if (action === 'del_cancel') {
      const delBtn = tr.querySelector('button[data-action="del"]');
      if (delBtn) {
        delBtn.textContent = delBtn.dataset.origText || 'del';
        delete delBtn.dataset.confirming;
      }
      btn.remove();
      return;
    }
  });

  /* ===========================
   * Enhancement #4: ADD MODAL
   * =========================== */

  function buildAddFieldList() {
    // Vertical form fields: omit actions and both ids; omit last_date/last_template (system-managed)
    const omit = new Set(['actions', 'id', 'last_date', 'last_template']);
    // We keep history/comment/grupo/status etc because your old add-row allowed them.
    // Admin-only: initials/private/wrong only appear for admin.
    const fields = COLS.filter(f => !omit.has(f));

    return fields.filter(f => {
      if (!IS_ADMIN && (f === 'private' || f === 'initials' || f === 'wrong')) return false;
      return true;
    });
  }

  function defaultValueForField(f) {
    if (f === 'status') return 'Prospect';
    if (f === 'MF') return '';
    if (f === 'grupo') return '';
    if (f === 'private' || f === 'wrong') return '0';
    if (f === 'initials') return ''; // admin may set; server will default if blank
    return '';
  }

  function makeInputForField(field, value) {
    // Field-specific widgets
    if (field === 'status') {
      const sel = document.createElement('select');
      const options = STATUS_OPTIONS.length ? STATUS_OPTIONS : [
        "Prospect","Email Sent","Follow-up","Meeting Booked","Qualified",
        "Proposal Sent","Negotiation","Closed won","Closed lost","Postponed"
      ];
      options.forEach(v => {
        const opt = document.createElement('option');
        opt.value = v;
        opt.textContent = v;
        if (v === value) opt.selected = true;
        sel.appendChild(opt);
      });
      return sel;
    }

    if (field === 'MF') {
      const sel = document.createElement('select');
      const opts = [
        {v:'', t:'(blank)'},
        {v:'M', t:'M'},
        {v:'F', t:'F'}
      ];
      opts.forEach(o => {
        const opt = document.createElement('option');
        opt.value = o.v;
        opt.textContent = o.t;
        if (o.v === value) opt.selected = true;
        sel.appendChild(opt);
      });
      return sel;
    }

    if (field === 'private' || field === 'wrong') {
      const wrap = document.createElement('div');
      wrap.className = 'tm-inlinechecks';

      const lab = document.createElement('label');
      const cb = document.createElement('input');
      cb.type = 'checkbox';
      cb.checked = String(value) === '1';
      lab.appendChild(cb);
      lab.appendChild(document.createTextNode(field));
      wrap.appendChild(lab);

      // Store field name for extraction
      cb.dataset.field = field;
      wrap.dataset.kind = 'checkbox';
      return wrap;
    }

    // Larger text areas for long fields
    if (field === 'comment' || field === 'history') {
      const ta = document.createElement('textarea');
      ta.value = value || '';
      return ta;
    }

    // Default text input
    const inp = document.createElement('input');
    inp.type = 'text';
    inp.value = value || '';
    return inp;
  }

  function openAddModal() {
    const fields = buildAddFieldList();

    const backdrop = document.createElement('div');
    backdrop.className = 'tm-backdrop';

    const modal = document.createElement('div');
    modal.className = 'tm-modal';

    const h = document.createElement('h3');
    h.textContent = 'Add company';
    modal.appendChild(h);

    const grid = document.createElement('div');
    grid.className = 'tm-grid';

    const controls = {}; // field -> element

    fields.forEach(field => {
      const box = document.createElement('div');
      box.className = 'tm-field';

      const label = document.createElement('label');
      label.textContent = field;
      box.appendChild(label);

      const val = defaultValueForField(field);

      const inputEl = makeInputForField(field, val);

      // Special case: initials should be editable for admin; otherwise not shown.
      // Still allow admin to leave blank (server will default to creator initials).
      if (field === 'initials') {
        if (inputEl.tagName === 'INPUT') {
          inputEl.placeholder = 'e.g. AB or ABC (admin only)';
        }
      }

      // Styling consistency
      if (inputEl.tagName === 'SELECT' || inputEl.tagName === 'INPUT' || inputEl.tagName === 'TEXTAREA') {
        inputEl.classList.add('search-input');
      }

      box.appendChild(inputEl);
      grid.appendChild(box);
      controls[field] = inputEl;
    });

    modal.appendChild(grid);

    const row = document.createElement('div');
    row.className = 'tm-row';

    const btnCancel = document.createElement('button');
    btnCancel.className = 'btn ghost';
    btnCancel.type = 'button';
    btnCancel.textContent = 'cancel';

    const btnSave = document.createElement('button');
    btnSave.className = 'btn primary';
    btnSave.type = 'button';
    btnSave.textContent = 'save';

    row.appendChild(btnCancel);
    row.appendChild(btnSave);
    modal.appendChild(row);

    function close() {
      document.removeEventListener('keydown', onKeyDown, true);
      backdrop.remove();
    }

    async function save() {
      const payload = {};

      fields.forEach(field => {
        const el = controls[field];
        if (!el) return;

        if (field === 'private' || field === 'wrong') {
          // checkbox wrapper
          const cb = el.querySelector('input[type="checkbox"]');
          payload[field] = cb && cb.checked ? '1' : '0';
          return;
        }

        if (field === 'status' || field === 'MF') {
          payload[field] = el.value;
          return;
        }

        payload[field] = (el.value || '').trim();
      });

      if (payload.linkedIn) payload.linkedIn = normalizeLinkedIn(payload.linkedIn);
      if (payload.website) payload.website = normalizeWebsite(payload.website);

      const resp = await fetch('list.php?action=add', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams(payload)
      });
      const out = await resp.json();
      if (!out.ok) {
        alert(out.error || 'add failed');
        return;
      }

      close();
      await fetchCountries();
      await fetchGrupos();
      await fetchRows();
    }

    function onKeyDown(e) {
      if (e.key === 'Escape') { e.preventDefault(); close(); }
      if (e.key === 'Enter' && (e.ctrlKey || e.metaKey)) { e.preventDefault(); save(); }
    }

    btnCancel.addEventListener('click', close);
    btnSave.addEventListener('click', save);

    backdrop.addEventListener('click', (e) => {
      if (e.target === backdrop) close();
    });

    document.addEventListener('keydown', onKeyDown, true);

    backdrop.appendChild(modal);
    document.body.appendChild(backdrop);

    // Focus first usable field
    const first = modal.querySelector('input, select, textarea');
    if (first) first.focus();
  }

  if (btnAdd) {
    btnAdd.addEventListener('click', () => openAddModal());
  }

  /* ===========================
   * Popup editor for long fields
   * =========================== */
  function openCellEditor(e) {
    e.preventDefault();
    e.stopPropagation();

    const td = e.currentTarget;
    const tr = td.closest('tr');
    if (!tr) return;

    const id = td.dataset.id;
    const field = td.dataset.field;
    if (!id || !field) return;

    const backdrop = document.createElement('div');
    backdrop.className = 'cell-editor-backdrop';

    const panel = document.createElement('div');
    panel.className = 'cell-editor';

    const header = document.createElement('div');
    header.className = 'cell-editor__header';

    const title = document.createElement('div');
    title.className = 'cell-editor__title';
    title.textContent = field + ' (id ' + id + ')';

    const actions = document.createElement('div');
    actions.className = 'cell-editor__actions';

    const btnCancel = document.createElement('button');
    btnCancel.className = 'btn small ghost';
    btnCancel.textContent = 'cancel';

    const btnCopy = document.createElement('button');
    btnCopy.className = 'btn small';
    btnCopy.textContent = 'copy';

    const btnPaste = document.createElement('button');
    btnPaste.className = 'btn small';
    btnPaste.textContent = 'paste';

    const btnSave = document.createElement('button');
    btnSave.className = 'btn small primary';
    btnSave.textContent = 'save';

    actions.appendChild(btnCancel);
    actions.appendChild(btnCopy);
    actions.appendChild(btnPaste);
    actions.appendChild(btnSave);

    header.appendChild(title);
    header.appendChild(actions);

    const ta = document.createElement('textarea');
    ta.className = 'cell-editor__textarea';
    ta.value = td.textContent || '';
    ta.readOnly = false;
    ta.disabled = false;

    panel.appendChild(header);
    panel.appendChild(ta);

    let onKeyDown = null;
    function close() {
      if (onKeyDown) document.removeEventListener('keydown', onKeyDown, true);
      backdrop.remove();
    }

    btnCancel.addEventListener('click', close);

    btnCopy.addEventListener('click', async () => {
      try { await navigator.clipboard.writeText(ta.value); } catch (_) {}
      close();
    });

    btnPaste.addEventListener('click', async () => {
      try {
        const text = await navigator.clipboard.readText();
        if (typeof text === 'string') ta.value = text;
      } catch (_) {}
      close();
    });

    btnSave.addEventListener('click', async () => {
      let newVal = ta.value;
      if (field === 'linkedIn') newVal = normalizeLinkedIn(newVal);
      if (field === 'website') newVal = normalizeWebsite(newVal);

      td.textContent = newVal;
      td.title = newVal;

      await postUpdate(id, field, newVal);
      close();
    });

    onKeyDown = function (e) {
      if (e.key === 'Escape') { e.preventDefault(); close(); }
    };
    document.addEventListener('keydown', onKeyDown, true);

    backdrop.addEventListener('click', (e) => {
      if (e.target === backdrop) close();
    });

    backdrop.appendChild(panel);
    document.body.appendChild(backdrop);

    setTimeout(() => {
      ta.focus();
      try { ta.select(); ta.setSelectionRange(0, ta.value.length); } catch (_) {}
    }, 0);
  }

  // Helpers
  function escapeHtml(s) {
    return String(s ?? '').replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#039;'}[m]));
  }
  function escapeAttr(s) {
    return escapeHtml(s).replace(/"/g, '&quot;');
  }

  /* ---------- Init ---------- */
  (async function init() {
    await fetchStatuses();
    await fetchCountries();
    await fetchGrupos();
    await fetchRows();
  })();
})();
