<?php
// install.php — TopMail installer / schema initializer (collision-safe)
//
// Purpose:
//  - Connects to DB server, creates the database (if missing)
//  - Creates/updates required tables: companies, templates, users, log
//  - Adds missing columns/indexes (best-effort migration)
//
// Usage:
//  1) Upload this file alongside your other PHP files.
//  2) Visit: /install.php (optionally add ?debug=1)
//  3) Delete install.php afterwards (recommended)

declare(strict_types=1);

$DEBUG = isset($_GET['debug']);

// -----------------------------------------------------------------------------
// IMPORTANT: set credentials here (do NOT include includes/db.php to avoid clashes)
// -----------------------------------------------------------------------------
$DB_HOST = 'localhost'; // change if needed
$DB_PORT = 3306;
$DB_NAME = 'visitormail_campaign';
$DB_USER = 'visitormail_dbf_admin';
$DB_PASS = 'Go2026&&clebex!!';

// -----------------------------------------------------------------------------
// Helpers (prefixed to avoid collisions)
// -----------------------------------------------------------------------------
function tm_install_h(string $s): string { return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

function tm_install_out(string $msg, bool $ok = true): void {
  $color = $ok ? '#1f7d58' : '#b33';
  echo "<div style=\"padding:10px;margin:8px 0;border:1px solid {$color};border-radius:10px;white-space:pre-wrap;\">"
     . tm_install_h($msg)
     . "</div>";
}

function tm_install_fail(string $msg, ?Throwable $e, bool $debug): void {
  tm_install_out($msg . ($debug && $e ? ("\n\n" . $e->getMessage()) : ''), false);
  exit;
}

function tm_install_pdo_server(string $host, int $port, string $user, string $pass): PDO {
  $dsn = "mysql:host={$host};port={$port};charset=utf8mb4";
  return new PDO($dsn, $user, $pass, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
}

function tm_install_pdo_db(string $host, int $port, string $db, string $user, string $pass): PDO {
  $dsn = "mysql:host={$host};port={$port};dbname={$db};charset=utf8mb4";
  return new PDO($dsn, $user, $pass, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
}

function tm_install_ensure_column(PDO $pdo, string $table, string $column, string $ddl): void {
  $stmt = $pdo->prepare("
    SELECT COUNT(*)
    FROM information_schema.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
      AND TABLE_NAME = :t
      AND COLUMN_NAME = :c
  ");
  $stmt->execute([':t' => $table, ':c' => $column]);
  $exists = (int)$stmt->fetchColumn() > 0;
  if (!$exists) {
    $pdo->exec("ALTER TABLE `$table` ADD COLUMN $ddl");
  }
}

function tm_install_ensure_index(PDO $pdo, string $table, string $indexName, string $ddlAddIndex): void {
  $stmt = $pdo->prepare("
    SELECT COUNT(*)
    FROM information_schema.STATISTICS
    WHERE TABLE_SCHEMA = DATABASE()
      AND TABLE_NAME = :t
      AND INDEX_NAME = :i
  ");
  $stmt->execute([':t'=>$table, ':i'=>$indexName]);
  $exists = (int)$stmt->fetchColumn() > 0;
  if (!$exists) {
    $pdo->exec("ALTER TABLE `$table` ADD $ddlAddIndex");
  }
}

// -----------------------------------------------------------------------------
// Output header
// -----------------------------------------------------------------------------
echo "<!doctype html><html><head><meta charset='utf-8'><meta name='viewport' content='width=device-width,initial-scale=1'>";
echo "<title>TopMail Install</title></head><body style='font:14px/1.4 system-ui,Segoe UI,Roboto,Arial;padding:18px;max-width:980px;margin:0 auto;'>";
echo "<h2>TopMail Installer</h2>";

tm_install_out(
  "Target DB: {$DB_HOST}:{$DB_PORT} / {$DB_NAME}\n".
  "User: {$DB_USER}\n".
  "Debug: " . ($DEBUG ? "ON" : "OFF")
);

// -----------------------------------------------------------------------------
// Connect to server, ensure DB, connect to DB
// -----------------------------------------------------------------------------
try {
  $server = tm_install_pdo_server($DB_HOST, (int)$DB_PORT, $DB_USER, $DB_PASS);
  tm_install_out("Connected to MySQL server.");
} catch (Throwable $e) {
  tm_install_fail("Cannot connect to MySQL server with provided credentials.", $e, $DEBUG);
}

try {
  $server->exec("CREATE DATABASE IF NOT EXISTS `{$DB_NAME}` CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci");
  tm_install_out("Database ensured: {$DB_NAME}");
} catch (Throwable $e) {
  tm_install_fail("Failed to create/ensure database: {$DB_NAME}", $e, $DEBUG);
}

try {
  $pdo = tm_install_pdo_db($DB_HOST, (int)$DB_PORT, $DB_NAME, $DB_USER, $DB_PASS);
  tm_install_out("Connected to database: {$DB_NAME}");
} catch (Throwable $e) {
  tm_install_fail("Cannot connect to database after creation: {$DB_NAME}", $e, $DEBUG);
}

// -----------------------------------------------------------------------------
// Create tables (idempotent)
// -----------------------------------------------------------------------------
try {
  $pdo->exec("
    CREATE TABLE IF NOT EXISTS companies (
      id INT AUTO_INCREMENT PRIMARY KEY,
      country TEXT,
      language CHAR(5),
      company_name TEXT,
      sector TEXT,
      website TEXT,
      ceo_name TEXT,
      MF CHAR(1) NULL,
      job TEXT,
      linkedIn TEXT,
      general_email TEXT,
      hq_city TEXT,
      email TEXT,
      wrong TINYINT(1),
      status TEXT,
      grupo TEXT,
      comment TEXT,
      last_date DATETIME NULL,
      last_template TEXT,
      history MEDIUMTEXT,
      initials TEXT(3),
      private TINYINT(1)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ");

  $pdo->exec("
    CREATE TABLE IF NOT EXISTS templates (
      id INT AUTO_INCREMENT PRIMARY KEY,
      ref TEXT,
      title1 TEXT,
      title2 TEXT,
      title3 TEXT,
      body MEDIUMTEXT,
      language CHAR(5)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ");

  $pdo->exec("
    CREATE TABLE IF NOT EXISTS users (
      id INT AUTO_INCREMENT PRIMARY KEY,
      name TEXT,
      initials TEXT(3),
      email TEXT,
      role TEXT,
      password_hash TEXT NULL,
      login_code TEXT NULL,
      code_expires DATETIME NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      UNIQUE KEY uniq_users_email (email(190)),
      UNIQUE KEY uniq_users_initials (initials(10))
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ");

  $pdo->exec("
    CREATE TABLE IF NOT EXISTS log (
      id INT AUTO_INCREMENT PRIMARY KEY,
      id_companies INT NOT NULL,
      id_templates INT NOT NULL,
      ref TEXT,
      language CHAR(5),
      title_nr TINYINT,
      timestamp TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      INDEX (id_companies),
      INDEX (id_templates)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ");

  tm_install_out("Tables ensured: companies, templates, users, log");
} catch (Throwable $e) {
  tm_install_fail("Failed creating base tables.", $e, $DEBUG);
}

// -----------------------------------------------------------------------------
// Best-effort migrations (add missing columns)
// -----------------------------------------------------------------------------
try {
  // companies
  tm_install_ensure_column($pdo, 'companies', 'job',      "job TEXT");
  tm_install_ensure_column($pdo, 'companies', 'wrong',    "wrong TINYINT(1)");
  tm_install_ensure_column($pdo, 'companies', 'status',   "status TEXT");
  tm_install_ensure_column($pdo, 'companies', 'grupo',    "grupo TEXT");
  tm_install_ensure_column($pdo, 'companies', 'initials', "initials TEXT(3)");
  tm_install_ensure_column($pdo, 'companies', 'private',  "private TINYINT(1)");

  // users
  tm_install_ensure_column($pdo, 'users', 'password_hash', "password_hash TEXT NULL");
  tm_install_ensure_column($pdo, 'users', 'login_code',    "login_code TEXT NULL");
  tm_install_ensure_column($pdo, 'users', 'code_expires',  "code_expires DATETIME NULL");
  tm_install_ensure_column($pdo, 'users', 'created_at',    "created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP");

  // log (in case older schema differs)
  tm_install_ensure_column($pdo, 'log', 'ref',      "ref TEXT");
  tm_install_ensure_column($pdo, 'log', 'language', "language CHAR(5)");
  tm_install_ensure_column($pdo, 'log', 'title_nr', "title_nr TINYINT");

  tm_install_out("Columns verified / migrated.");
} catch (Throwable $e) {
  tm_install_fail("Failed applying schema migration.", $e, $DEBUG);
}

// -----------------------------------------------------------------------------
// Ensure indexes (uniques are best-effort; may fail if duplicates exist)
// -----------------------------------------------------------------------------
try {
  try { tm_install_ensure_index($pdo, 'users', 'uniq_users_email', "UNIQUE KEY uniq_users_email (email(190))"); } catch (Throwable $e) {}
  try { tm_install_ensure_index($pdo, 'users', 'uniq_users_initials', "UNIQUE KEY uniq_users_initials (initials(10))"); } catch (Throwable $e) {}
  tm_install_ensure_index($pdo, 'log', 'id_companies', "INDEX (id_companies)");
  tm_install_ensure_index($pdo, 'log', 'id_templates', "INDEX (id_templates)");

  tm_install_out("Indexes ensured (best-effort for uniques if duplicates exist).");
} catch (Throwable $e) {
  tm_install_fail("Failed ensuring indexes.", $e, $DEBUG);
}

// -----------------------------------------------------------------------------
// Summary
// -----------------------------------------------------------------------------
try {
  $cntCompanies = (int)$pdo->query("SELECT COUNT(*) FROM companies")->fetchColumn();
  $cntTemplates = (int)$pdo->query("SELECT COUNT(*) FROM templates")->fetchColumn();
  $cntUsers     = (int)$pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
  $cntLog       = (int)$pdo->query("SELECT COUNT(*) FROM log")->fetchColumn();

  tm_install_out(
    "Install complete.\n\nRow counts:\n".
    "- companies: {$cntCompanies}\n".
    "- templates: {$cntTemplates}\n".
    "- users: {$cntUsers}\n".
    "- log: {$cntLog}"
  );
} catch (Throwable $e) {
  tm_install_out("Install complete, but could not read row counts.", false);
}

echo "<hr><p><strong>Next steps</strong></p>";
echo "<ol>";
echo "<li>Go to <code>login.php</code> and enter an email to create the first user (becomes <code>admin</code>).</li>";
echo "<li>After login, manage users via <code>users.php</code> (admin only).</li>";
echo "<li><strong>Delete install.php</strong> from the server once confirmed.</li>";
echo "</ol>";

echo "</body></html>";
