<?php
require_once __DIR__ . '/includes/auth.php';

require_login();
$me = current_user();
$is_admin = ($me && ($me['role'] ?? '') === 'admin');

/* ---------- Safe helpers (guarded) ---------- */
if (!function_exists('json_out')) {
  function json_out($data, $code = 200) {
    if (!headers_sent()) {
      http_response_code($code);
      header('Content-Type: application/json; charset=utf-8');
    }
    echo json_encode($data);
    exit;
  }
}
if (!function_exists('sanitize_sort')) {
  function sanitize_sort($field, $allowed) {
    return in_array($field, $allowed, true) ? $field : 'id';
  }
}

/* ---------- Visibility clause for companies ---------- */
[$visClause, $visParams] = companies_visibility_clause($me);

/* ---------- AJAX endpoints ---------- */
if (!empty($_GET['action'])) {
  try {
    $action = $_GET['action'];

    if ($action === 'countries') {
      $stmt = $pdo->prepare("SELECT DISTINCT IFNULL(country,'') AS country FROM companies WHERE $visClause ORDER BY country ASC");
      $stmt->execute($visParams);
      $rows = $stmt->fetchAll();
      $countries = [];
      foreach ($rows as $r) { $countries[] = ($r['country'] !== '' ? $r['country'] : '(blank)'); }
      json_out(['countries' => $countries]);
    }

    if ($action === 'grupos') {
      $stmt = $pdo->prepare("SELECT DISTINCT IFNULL(grupo,'') AS grupo FROM companies WHERE $visClause ORDER BY grupo ASC");
      $stmt->execute($visParams);
      $rows = $stmt->fetchAll();
      $grupos = [];
      foreach ($rows as $r) { $grupos[] = ($r['grupo'] !== '' ? $r['grupo'] : '(blank)'); }
      json_out(['grupos' => $grupos]);
    }

    if ($action === 'list') {
      $country  = $_GET['country'] ?? 'all';
      $grupo    = $_GET['grupo'] ?? 'all';
      $q        = trim((string)($_GET['q'] ?? ''));

      $sortable = [
        'id','company_name','ceo_name','MF','linkedIn','last_date','last_template',
        'status','grupo','initials','private','wrong',
        'country','language','history','email','sector','website','general_email','hq_city','comment'
      ];
      $sort = sanitize_sort($_GET['sort'] ?? 'id', $sortable);
      $dir  = (isset($_GET['dir']) && strtoupper($_GET['dir']) === 'DESC') ? 'DESC' : 'ASC';

      $where = [];
      $params = $visParams;

      $where[] = "($visClause)";

      if ($country && $country !== 'all') { $where[] = "country = :c"; $params[':c'] = $country; }
      if ($grupo && $grupo !== 'all') { $where[] = "grupo = :g"; $params[':g'] = $grupo; }

      if ($q !== '') {
        $params[':q'] = "%$q%";
        $where[] = "("
          . "CAST(id AS CHAR) LIKE :q OR country LIKE :q OR language LIKE :q OR company_name LIKE :q OR sector LIKE :q OR website LIKE :q OR "
          . "ceo_name LIKE :q OR MF LIKE :q OR linkedIn LIKE :q OR general_email LIKE :q OR hq_city LIKE :q OR "
          . "email LIKE :q OR comment LIKE :q OR last_template LIKE :q OR history LIKE :q OR "
          . "status LIKE :q OR grupo LIKE :q OR initials LIKE :q OR "
          . "DATE_FORMAT(last_date, '%Y-%m-%d %H:%i:%s') LIKE :q"
          . ")";
      }

      $sql = "SELECT * FROM companies";
      if ($where) $sql .= " WHERE " . implode(" AND ", $where);
      $sql .= " ORDER BY $sort $dir";

      $stmt = $pdo->prepare($sql);
      $stmt->execute($params);
      $rows = $stmt->fetchAll();

      $countSql = "SELECT COUNT(*) FROM companies";
      if ($where) $countSql .= " WHERE " . implode(" AND ", $where);
      $countStmt = $pdo->prepare($countSql);
      $countStmt->execute($params);
      $total = (int)$countStmt->fetchColumn();

      json_out(['rows' => $rows, 'total' => $total, 'is_admin' => $is_admin]);
    }

    if ($action === 'update' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      $id    = (int)($_POST['id'] ?? 0);
      $field = (string)($_POST['field'] ?? '');
      $value = (string)($_POST['value'] ?? '');

      if (!$id) json_out(['ok'=>false,'error'=>'missing id'], 400);

      $st = $pdo->prepare("SELECT * FROM companies WHERE id = :id LIMIT 1");
      $st->execute([':id'=>$id]);
      $row = $st->fetch();
      if (!$row) json_out(['ok'=>false,'error'=>'not found'], 404);
      if (!can_access_company_row($row, $me)) json_out(['ok'=>false,'error'=>'forbidden'], 403);

      $allowed = [
        'country','language','company_name','sector','website','ceo_name','MF','linkedIn',
        'general_email','hq_city','email','comment','last_date','last_template','history',
        'status','grupo','wrong'
      ];

      if ($field === 'private' || $field === 'initials') {
        if (!$is_admin) json_out(['ok'=>false,'error'=>'forbidden'], 403);
        $allowed[] = 'private';
        $allowed[] = 'initials';
      }

      if (!in_array($field, $allowed, true)) json_out(['ok'=>false,'error'=>'invalid field'], 400);

      if ($field === 'wrong' || $field === 'private') {
        $value = ($value === '1' || strtolower($value) === 'true' || strtoupper($value) === 'Y') ? '1' : '0';
      }

      $pdo->prepare("UPDATE companies SET `$field` = :v WHERE id = :id")
          ->execute([':v'=>$value, ':id'=>$id]);

      json_out(['ok'=>true]);
    }

    if ($action === 'add' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      $creator_initials = (string)($me['initials'] ?? '');
      if ($creator_initials === '') json_out(['ok'=>false,'error'=>'user has no initials'], 400);

      // Fields accepted from form (public)
      $f = [
        'country','language','company_name','sector','website','ceo_name','MF','linkedIn',
        'general_email','hq_city','email','comment','grupo','status','history'
      ];
      $vals = [];
      foreach ($f as $k) $vals[$k] = $_POST[$k] ?? null;

      // Defaults
      if (!$vals['status']) $vals['status'] = 'Prospect';
      if ($vals['grupo'] === '') $vals['grupo'] = null;
      if ($vals['history'] === '') $vals['history'] = null;

      // Admin-only inputs on add
      $initials = $creator_initials;
      $private = 0;
      $wrong = 0;

      if ($is_admin) {
        if (isset($_POST['initials']) && trim((string)$_POST['initials']) !== '') {
          $initials = strtoupper(trim((string)$_POST['initials']));
        }
        if (isset($_POST['private'])) $private = (($_POST['private'] === '1' || strtoupper($_POST['private']) === 'Y') ? 1 : 0);
        if (isset($_POST['wrong'])) $wrong = (($_POST['wrong'] === '1' || strtoupper($_POST['wrong']) === 'Y') ? 1 : 0);
      }

      $stmt = $pdo->prepare("
        INSERT INTO companies (
          country,language,company_name,sector,website,ceo_name,MF,linkedIn,general_email,hq_city,email,comment,
          status, grupo, initials, private, wrong,
          last_date,last_template,history
        )
        VALUES (
          :country,:language,:company_name,:sector,:website,:ceo_name,:MF,:linkedIn,:general_email,:hq_city,:email,:comment,
          :status,:grupo,:initials,:private,:wrong,
          NULL,NULL,:history
        )
      ");

      $vals['initials'] = $initials;
      $vals['private'] = $private;
      $vals['wrong'] = $wrong;

      $stmt->execute($vals);

      json_out(['ok'=>true, 'id'=>$pdo->lastInsertId()]);
    }

    if ($action === 'delete' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      $id = (int)($_POST['id'] ?? 0);
      if (!$id) json_out(['ok'=>false,'error'=>'missing id'], 400);
      if (!$is_admin) json_out(['ok'=>false,'error'=>'forbidden'], 403);

      $pdo->prepare("DELETE FROM companies WHERE id = :id")->execute([':id'=>$id]);
      json_out(['ok'=>true]);
    }

    json_out(['ok'=>false,'error'=>'unknown action'], 400);
  } catch (Throwable $e) {
    json_out(['ok'=>false, 'error'=>'server', 'detail'=>$e->getMessage()], 500);
  }
}

/* ---------- Normal page render ---------- */
include __DIR__ . '/header.php';
?>
<script>
  window.TOPMAIL_IS_ADMIN = <?= $is_admin ? 'true' : 'false' ?>;
</script>

<section>
  <div class="toolbar">
    <button id="btnFilter" class="btn">filter</button>
    <div id="filterPanel" class="dropdown hidden" style="display:flex;gap:8px;align-items:center;flex-wrap:wrap;">
      <select id="countryFilter"></select>
      <select id="grupoFilter"></select>
      <button id="applyFilter" class="btn small">apply</button>
      <button id="clearFilter" class="btn ghost small">all</button>
    </div>

    <button id="btnAdd" class="btn primary">add</button>

    <div class="search-wrap">
      <input id="q" class="search-input" type="text" placeholder="search all fields..." autocomplete="off">
      <button id="btnClearSearch" class="btn small ghost" title="clear">×</button>
    </div>
  </div>

  <div class="table-wrap">
    <table id="companiesTable">
      <thead>
        <tr>
          <?php
            $cols = [
              'actions','id','company_name','ceo_name','MF','linkedIn','last_date','last_template',
              'status','grupo','initials'
            ];
            if ($is_admin) $cols[] = 'private';
            $cols = array_merge($cols, [
              'wrong','country','language','history','email','sector','website','general_email','hq_city','comment','id'
            ]);

            foreach ($cols as $c) {
              $label = htmlspecialchars($c);
              $cls = ($c === 'actions') ? '' : 'sortable';
              echo "<th data-field=\"{$label}\" class=\"{$cls}\">{$label}</th>";
            }
          ?>
        </tr>
      </thead>
      <tbody></tbody>
    </table>
  </div>
</section>

<?php include __DIR__ . '/footer.php'; ?>
