<?php
require_once __DIR__ . '/includes/auth.php';

$next  = $_GET['next'] ?? 'list.php';
$error = '';
$info  = '';

if (is_logged_in()) {
  header('Location: ' . $next);
  exit;
}

$step = $_POST['step'] ?? 'email';

function users_count(): int {
  global $pdo;
  return (int)$pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
}

function fetch_user_by_email(string $email): ?array {
  global $pdo;
  $st = $pdo->prepare("SELECT * FROM users WHERE email = :e LIMIT 1");
  $st->execute([':e'=>$email]);
  $u = $st->fetch();
  return $u ?: null;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  try {
    if ($step === 'email') {
      $email = normalize_email($_POST['email'] ?? '');
      if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new RuntimeException("Please enter a valid email.");
      }

      $count = users_count();
      $bootstrap = ($count === 0);

      $u = fetch_user_by_email($email);

      if ($bootstrap) {
        // First user: create admin on the fly, then set password (no email, no code)
        if (!$u) {
          $pdo->prepare("INSERT INTO users (email, role, name, initials, password_hash) VALUES (:e, 'admin', NULL, NULL, NULL)")
              ->execute([':e'=>$email]);
          $u = fetch_user_by_email($email);
        } else {
          // If DB already has a user but we're still in bootstrap mode, ensure admin role
          $pdo->prepare("UPDATE users SET role='admin' WHERE id=:id")->execute([':id'=>(int)$u['id']]);
          $u = fetch_user_by_email($email);
        }

        $step = 'set_password';
        $_POST['email'] = $email;
        $info = "Create the admin password.";
      } else {
        // Not bootstrap: user must exist already
        if (!$u) {
          throw new RuntimeException("This email is not registered. Please ask an admin to create your user in Users.");
        }

        // If password exists -> login with password
        if (!empty($u['password_hash'])) {
          $step = 'password';
          $_POST['email'] = $email;
          $info = "Enter your password.";
        } else {
          // First login (or admin reset): set a new password
          $step = 'set_password';
          $_POST['email'] = $email;
          $info = "Set your password.";
        }
      }
    }

    elseif ($step === 'password') {
      $email = normalize_email($_POST['email'] ?? '');
      $pass  = (string)($_POST['password'] ?? '');

      $u = fetch_user_by_email($email);
      if (!$u || empty($u['password_hash'])) {
        throw new RuntimeException("Password login is not available for this account. Please set a password.");
      }
      if (!password_verify($pass, $u['password_hash'])) {
        throw new RuntimeException("Invalid password.");
      }

      $_SESSION['topmail_user_id'] = (int)$u['id'];
      header('Location: ' . $next);
      exit;
    }

    elseif ($step === 'set_password') {
      $email = normalize_email($_POST['email'] ?? '');
      $p1 = (string)($_POST['password1'] ?? '');
      $p2 = (string)($_POST['password2'] ?? '');

      if ($p1 === '' || $p2 === '') throw new RuntimeException("Password and confirmation are required.");
      if ($p1 !== $p2) throw new RuntimeException("Passwords do not match.");
      if (strlen($p1) < 8) throw new RuntimeException("Password must be at least 8 characters.");

      $u = fetch_user_by_email($email);

      // Bootstrap safety: if somehow not found, only allow if still empty table
      if (!$u) {
        if (users_count() !== 0) {
          throw new RuntimeException("This email is not registered. Please ask an admin to create your user in Users.");
        }
        $pdo->prepare("INSERT INTO users (email, role, name, initials, password_hash) VALUES (:e, 'admin', NULL, NULL, NULL)")
            ->execute([':e'=>$email]);
        $u = fetch_user_by_email($email);
      }

      $hash = password_hash($p1, PASSWORD_DEFAULT);
      $pdo->prepare("UPDATE users SET password_hash = :h WHERE id = :id")->execute([':h'=>$hash, ':id'=>(int)$u['id']]);

      // If name/initials not set yet, we require setup (mandatory) after password set
      if (empty($u['name']) || empty($u['initials'])) {
        $step = 'setup_profile';
        $_POST['email'] = $email;
        $info = "Complete your profile.";
      } else {
        $_SESSION['topmail_user_id'] = (int)$u['id'];
        header('Location: ' . $next);
        exit;
      }
    }

    elseif ($step === 'setup_profile') {
      $email = normalize_email($_POST['email'] ?? '');
      $name = trim((string)($_POST['name'] ?? ''));
      $initials = normalize_initials($_POST['initials'] ?? '');

      if ($name === '') throw new RuntimeException("Name is required.");
      if (!initials_valid($initials)) throw new RuntimeException("Initials must be 2–3 capital letters (A–Z).");

      $u = fetch_user_by_email($email);
      if (!$u) throw new RuntimeException("Unknown user.");

      // Enforce uniqueness of initials
      $st = $pdo->prepare("SELECT COUNT(*) FROM users WHERE initials = :i AND id <> :id");
      $st->execute([':i'=>$initials, ':id'=>(int)$u['id']]);
      if ((int)$st->fetchColumn() > 0) throw new RuntimeException("Initials already exist.");

      $pdo->prepare("UPDATE users SET name = :n, initials = :i WHERE id = :id")
          ->execute([':n'=>$name, ':i'=>$initials, ':id'=>(int)$u['id']]);

      $_SESSION['topmail_user_id'] = (int)$u['id'];
      header('Location: ' . $next);
      exit;
    }

  } catch (Throwable $e) {
    $error = $e->getMessage();
  }
}

include __DIR__ . '/header.php';
?>
<section style="max-width:680px">
  <h2>Login</h2>

  <?php if ($info): ?>
    <p class="note"><?= htmlspecialchars($info) ?></p>
  <?php endif; ?>
  <?php if ($error): ?>
    <p class="note" style="border-color:#b33;"><?= htmlspecialchars($error) ?></p>
  <?php endif; ?>

  <?php if ($step === 'email'): ?>
    <form method="post" class="card" autocomplete="off">
      <input type="hidden" name="step" value="email">
      <label>Email
        <input class="search-input" style="width:100%;max-width:420px" type="email" name="email" required>
      </label>
      <div style="margin-top:10px">
        <button class="btn primary">continue</button>
      </div>
    </form>

  <?php elseif ($step === 'password'): ?>
    <form method="post" class="card" autocomplete="off">
      <input type="hidden" name="step" value="password">
      <input type="hidden" name="email" value="<?= htmlspecialchars($_POST['email'] ?? '') ?>">
      <label>Email
        <input class="search-input" style="width:100%;max-width:420px" type="email" value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" readonly>
      </label>
      <label style="display:block;margin-top:10px">Password
        <input class="search-input" style="width:100%;max-width:420px" type="password" name="password" required>
      </label>
      <div style="margin-top:10px">
        <button class="btn primary">login</button>
        <a class="btn" href="login.php">back</a>
      </div>
    </form>

  <?php elseif ($step === 'set_password'): ?>
    <form method="post" class="card" autocomplete="off">
      <input type="hidden" name="step" value="set_password">
      <input type="hidden" name="email" value="<?= htmlspecialchars($_POST['email'] ?? '') ?>">

      <label>Email
        <input class="search-input" style="width:100%;max-width:420px" type="email" value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" readonly>
      </label>

      <label style="display:block;margin-top:10px">New password
        <input class="search-input" style="width:100%;max-width:420px" type="password" name="password1" required>
      </label>
      <label style="display:block;margin-top:10px">Confirm password
        <input class="search-input" style="width:100%;max-width:420px" type="password" name="password2" required>
      </label>

      <div style="margin-top:10px">
        <button class="btn primary">save password</button>
      </div>
    </form>

  <?php elseif ($step === 'setup_profile'): ?>
    <form method="post" class="card" autocomplete="off">
      <input type="hidden" name="step" value="setup_profile">
      <input type="hidden" name="email" value="<?= htmlspecialchars($_POST['email'] ?? '') ?>">

      <label>Email
        <input class="search-input" style="width:100%;max-width:420px" type="email" value="<?= htmlspecialchars($_POST['email'] ?? '') ?>" readonly>
      </label>

      <label style="display:block;margin-top:10px">Name
        <input class="search-input" style="width:100%;max-width:420px" type="text" name="name" required>
      </label>

      <label style="display:block;margin-top:10px">Initials (2–3 capital letters)
        <input class="search-input" style="width:100%;max-width:120px" type="text" name="initials" maxlength="3" required>
      </label>

      <div style="margin-top:10px">
        <button class="btn primary">finish</button>
      </div>
    </form>
  <?php endif; ?>
</section>

<?php include __DIR__ . '/footer.php'; ?>
