<?php
require_once __DIR__ . '/includes/auth.php';

require_login();
$me = current_user();
if (!$me || ($me['role'] ?? '') !== 'admin') {
  http_response_code(403);
  echo "Forbidden";
  exit;
}

$err = '';
$info = '';

if (!empty($_GET['action'])) {
  try {
    $action = (string)$_GET['action'];

    if ($action === 'list') {
      $rows = $pdo->query("SELECT id, name, initials, email, role, password_hash, created_at FROM users ORDER BY id ASC")->fetchAll();
      json_out(['rows'=>$rows]);
    }

    if ($action === 'create' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      $email = normalize_email($_POST['email'] ?? '');
      $name = trim((string)($_POST['name'] ?? ''));
      $initials = normalize_initials($_POST['initials'] ?? '');
      $role = (string)($_POST['role'] ?? 'user');

      if (!filter_var($email, FILTER_VALIDATE_EMAIL)) json_out(['ok'=>false,'error'=>'invalid email'], 400);
      if ($name === '') json_out(['ok'=>false,'error'=>'name required'], 400);
      if (!initials_valid($initials)) json_out(['ok'=>false,'error'=>'initials must be 2–3 capital letters'], 400);
      if (!in_array($role, ['admin','manager','user'], true)) $role = 'user';

      // enforce uniqueness
      $st = $pdo->prepare("SELECT COUNT(*) FROM users WHERE email = :e");
      $st->execute([':e'=>$email]);
      if ((int)$st->fetchColumn() > 0) json_out(['ok'=>false,'error'=>'email exists'], 400);

      $st = $pdo->prepare("SELECT COUNT(*) FROM users WHERE initials = :i");
      $st->execute([':i'=>$initials]);
      if ((int)$st->fetchColumn() > 0) json_out(['ok'=>false,'error'=>'initials exists'], 400);

      $pdo->prepare("INSERT INTO users (name, initials, email, role, password_hash) VALUES (:n,:i,:e,:r,NULL)")
          ->execute([':n'=>$name, ':i'=>$initials, ':e'=>$email, ':r'=>$role]);

      json_out(['ok'=>true]);
    }

    if ($action === 'set_role' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      $id = (int)($_POST['id'] ?? 0);
      $role = (string)($_POST['role'] ?? 'user');
      if (!$id) json_out(['ok'=>false,'error'=>'missing id'], 400);
      if (!in_array($role, ['admin','manager','user'], true)) json_out(['ok'=>false,'error'=>'invalid role'], 400);

      $pdo->prepare("UPDATE users SET role = :r WHERE id = :id")->execute([':r'=>$role, ':id'=>$id]);
      json_out(['ok'=>true]);
    }

    if ($action === 'reset_password' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      $id = (int)($_POST['id'] ?? 0);
      if (!$id) json_out(['ok'=>false,'error'=>'missing id'], 400);

      // Prevent admin from locking themselves out accidentally (optional safety)
      if ($id === (int)$me['id']) {
        json_out(['ok'=>false,'error'=>'cannot reset your own password here'], 400);
      }

      $pdo->prepare("UPDATE users SET password_hash = NULL WHERE id = :id")->execute([':id'=>$id]);
      json_out(['ok'=>true]);
    }

    json_out(['ok'=>false,'error'=>'unknown action'], 400);
  } catch (Throwable $e) {
    json_out(['ok'=>false,'error'=>'server','detail'=>$e->getMessage()], 500);
  }
}

include __DIR__ . '/header.php';
?>
<section>
  <h2>Users</h2>

  <div class="card" style="margin-bottom:14px">
    <h3 style="margin:0 0 10px 0">Create user</h3>
    <form id="createUserForm">
      <div style="display:flex;gap:10px;flex-wrap:wrap;align-items:flex-end">
        <label>Email<br><input class="search-input" type="email" name="email" required></label>
        <label>Name<br><input class="search-input" type="text" name="name" required></label>
        <label>Initials<br><input class="search-input" type="text" name="initials" maxlength="3" required></label>
        <label>Role<br>
          <select class="search-input" name="role">
            <option value="user">user</option>
            <option value="manager">manager</option>
            <option value="admin">admin</option>
          </select>
        </label>
        <button class="btn primary" type="submit">create</button>
      </div>
      <p class="note" style="margin-top:10px">
        Password is set by the user on first login. To force a reset, use “reset password”.
      </p>
    </form>
  </div>

  <div class="table-wrap">
    <table id="usersTable">
      <thead>
        <tr>
          <th>id</th>
          <th>email</th>
          <th>name</th>
          <th>initials</th>
          <th>role</th>
          <th>password</th>
          <th>created</th>
          <th>actions</th>
        </tr>
      </thead>
      <tbody></tbody>
    </table>
  </div>
</section>

<script>
(async function(){
  const tableBody = document.querySelector('#usersTable tbody');
  const form = document.getElementById('createUserForm');

  async function loadUsers(){
    const res = await fetch('users.php?action=list');
    const data = await res.json();
    tableBody.innerHTML = '';
    (data.rows || []).forEach(r => {
      const tr = document.createElement('tr');
      const hasPw = r.password_hash ? 'set' : 'not set';
      tr.innerHTML = `
        <td>${r.id}</td>
        <td>${escapeHtml(r.email||'')}</td>
        <td>${escapeHtml(r.name||'')}</td>
        <td>${escapeHtml(r.initials||'')}</td>
        <td>
          <select data-role="${r.id}">
            <option value="user" ${r.role==='user'?'selected':''}>user</option>
            <option value="manager" ${r.role==='manager'?'selected':''}>manager</option>
            <option value="admin" ${r.role==='admin'?'selected':''}>admin</option>
          </select>
        </td>
        <td>${hasPw}</td>
        <td>${escapeHtml(r.created_at||'')}</td>
        <td>
          <button class="btn small" data-reset="${r.id}">reset password</button>
        </td>
      `;
      tableBody.appendChild(tr);
    });

    // role change
    document.querySelectorAll('select[data-role]').forEach(sel => {
      sel.addEventListener('change', async () => {
        const id = sel.getAttribute('data-role');
        const role = sel.value;
        const fd = new FormData();
        fd.append('id', id);
        fd.append('role', role);
        const rr = await fetch('users.php?action=set_role', { method:'POST', body: fd });
        const dd = await rr.json();
        if (!dd.ok) alert(dd.error || 'role update failed');
      });
    });

    // reset password
    document.querySelectorAll('button[data-reset]').forEach(btn => {
      btn.addEventListener('click', async () => {
        const id = btn.getAttribute('data-reset');
        if (!confirm('Clear password for this user? They will set a new password on next login.')) return;
        const fd = new FormData();
        fd.append('id', id);
        const rr = await fetch('users.php?action=reset_password', { method:'POST', body: fd });
        const dd = await rr.json();
        if (!dd.ok) alert(dd.error || 'reset failed');
        await loadUsers();
      });
    });
  }

  function escapeHtml(s){
    return String(s).replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#039;'}[m]));
  }

  form.addEventListener('submit', async (e) => {
    e.preventDefault();
    const fd = new FormData(form);
    const rr = await fetch('users.php?action=create', { method:'POST', body: fd });
    const dd = await rr.json();
    if (!dd.ok) alert(dd.error || 'create failed');
    else form.reset();
    await loadUsers();
  });

  await loadUsers();
})();
</script>

<?php include __DIR__ . '/footer.php'; ?>
